/**
  ******************************************************************************
  * @file    rwlock.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   Implements several variations of a reader writer lock
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 

// This code accompanies chapter 6 of the book "Embedded Controller - Grundlagen und praktische Umsetzung fr industrielle Anwendungen" by Rdiger R. Asche.
// You should really read the chapter before working with this code. It basically demonstrates several (both correct and incorrect) implemenations of a
// reader writer lock to compare the solutions against each other.

/* Includes ------------------------------------------------------------------*/

#include "main.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

//#define SECURE_IMPLEMENTATION
#define USE_BUILTINS
//#define PROVOKE_RACECONDITION
#define BUMP_WRITER_PRIORITY

typedef struct RWLockManagement
{
    xSemaphoreHandle    m_GroupSeparator;
    signed long         m_CurrentReaderCount;
#ifdef SECURE_IMPLEMENTATION
    xSemaphoreHandle    m_CounterProtector;
#endif
    volatile unsigned long     m_ReaderSpinLock;
#ifdef BUMP_WRITER_PRIORITY
    unsigned long     m_WriterPending;
#endif
} RWLockManagement, *PRWLockManagement;

RWLockManagement g_SLM;

/** @brief Implements an atomic increment and test for 0
 *
 *  using either of the preprocessor definitions SECURE_IMPLEMENTATION, USE_BUILTINS or PROVOKE_RACECONDITION, you can test several possible implementations.
 *  @param p_Lock Pointer to a lock object
 *  @return true if the lock count is 0 AFTER the increment, false otherwise
 */

unsigned short INCREMENTTEST0(PRWLockManagement p_Lock)
{
#ifdef SECURE_IMPLEMENTATION
    unsigned short aResult;
    xSemaphoreTake(p_Lock->m_CounterProtector, portMAX_DELAY);
    aResult = (++p_Lock->m_CurrentReaderCount == 0);
    xSemaphoreGive(p_Lock->m_CounterProtector);
    return aResult;
#endif
#ifdef PROVOKE_RACECONDITION
    volatile signed long aDum = p_Lock->m_CurrentReaderCount;
  {
      volatile unsigned long aCPUGuzzler = 0;
      while (aCPUGuzzler++ < 200);
  }
    p_Lock->m_CurrentReaderCount = aDum + 1;
    return (p_Lock->m_CurrentReaderCount == 0);    
#endif
#ifdef USE_BUILTINS
    return (__sync_add_and_fetch(&p_Lock->m_CurrentReaderCount,1) == 0);
#endif
}

/** @brief Implements an atomic decrement and test for negative
 *
 *  using either of the preprocessor definitions SECURE_IMPLEMENTATION, USE_BUILTINS or PROVOKE_RACECONDITION, you can test several possible implementations.
 *  @param p_Lock Pointer to a lock object
 *  @return true if the lock count is negative AFTER the decrement, false otherwise
 */

unsigned short DECREMENTTESTNEG(PRWLockManagement p_Lock) 
{
#ifdef SECURE_IMPLEMENTATION
    unsigned short aResult;
    xSemaphoreTake(p_Lock->m_CounterProtector, portMAX_DELAY);
    aResult = (--(p_Lock->m_CurrentReaderCount) < 0);
    xSemaphoreGive(p_Lock->m_CounterProtector);
    return aResult;
#endif
#ifdef PROVOKE_RACECONDITION
    volatile signed long aDum = p_Lock->m_CurrentReaderCount;
  {
      volatile unsigned long aCPUGuzzler = 0;
      while (aCPUGuzzler++ < 20);
  }
//    vTaskDelay(1);
    p_Lock->m_CurrentReaderCount = aDum - 1;
    return (p_Lock->m_CurrentReaderCount < 0);    
#endif
#ifdef USE_BUILTINS
    return (__sync_sub_and_fetch(&p_Lock->m_CurrentReaderCount,1) < 0);
#endif
}

/** @brief Initializes a reader writer lock object
 *
 *  using either of the preprocessor definitions SECURE_IMPLEMENTATION, USE_BUILTINS or PROVOKE_RACECONDITION, you can test several possible implementations.
 *  @param p_Lock Pointer to a lock object
 *  @return true if the lock object could be created
 */

unsigned long CreateRwLock(PRWLockManagement p_Lock)
{
    unsigned long a_Result = 0;
  // wir benutzen counting semaphoren, weil claim und release durch verschiedene tasks erfolgen knnen mssen (also im Reader Fall vom 1. und letzten reader, die verschieden sein
    // knnen. Das geht allerdings bei counting oder binren Semaphoren (FreeRTOS fhrt nur bei muteces Buch ber ownership).
#ifdef USE_COUNTING_SEMAPHORE
    p_Lock->m_GroupSeparator = xSemaphoreCreateCounting(1,1);
#else
    vSemaphoreCreateBinary(p_Lock->m_GroupSeparator);
#endif
    // Dieser mutex ist dazu da, sicherzustellen da der Zugriff auf den Reader counter atomisch verluft, sonst besteht deadlock Gefahr...
    if (p_Lock->m_GroupSeparator)
    {
        p_Lock->m_CurrentReaderCount = -1;
        p_Lock->m_ReaderSpinLock = 0;
#ifdef SECURE_IMPLEMENTATION
        p_Lock->m_CounterProtector = xSemaphoreCreateMutex();
        if (p_Lock->m_CounterProtector)
        {
            a_Result = 1;
        }
#else
	// Beide Varianten erzeugen exact dasselbe Objekt! Der Code zum Erzeugen einer binren Sempahore wre wie folgt:
#ifndef USE_COUNTING_SEMAPHORE
        if (xSemaphoreGive(p_Lock->m_GroupSeparator) == pdTRUE) // einmal geben, weil die binre Semaphore geschlossen erzeugt wird
#endif        
        a_Result = 1;
#endif
    }
    return a_Result;
}

/** @brief Helper function to generate a lock object without parameters
 *
 *  @return none (assumes success)
 */

void InitGlobalRWLock(void)
{
    CreateRwLock(&g_SLM);
}

/** @brief Implements an RW lock claim. Internal function so the caller won't need to deal with data structures. 
 *
 *  @param p_Lock Pointer to a lock object
 *  @param p_Mode: LOCK_WRITER or LOCK_READER
 *  @return none (blocks until the lock becomes or is available for the caller's purpose)
 */

void ClaimRwLockInternal(PRWLockManagement p_Lock,unsigned short p_Mode)
{
    if (p_Mode == LOCK_WRITER)
    {
        // Dies ist ein hllenfieses Ding. Eigentlich mte der Reader die Semaphore
        // kriegen sobald der letzte Reader sie freigegeben hat, weil er seine Prioritt
        // temporr erhht, bevor er dran kommt. Leider aber passiert das nicht, d.h. die
        // Reader geben sich die Semaphore immer gleichzeitig ab und hungern den Reader aus.
        // Im ersten Versuch haben wir deswegen eine Blocksignatur gehabt, die der Writer dazu
        // benutzt, die Readers aus dem lock rauszuhalten. Leider aber gibt es Flle, in denen
        // Reader den lock rekursiv aufrufen, d.h. wenn der Writer dazwischen kommt, legt  sich ein
        // Reader selber lahm - deadlock! Da wrde TLS praktisch werden, das haben wir aber nicht.
        // Beim herausgehen suspendieren ist auch keine Lsung, da wir dann am Ende des rekursiven claimens
        // hngenbleiben. Der einzige Weg heraus besteht darin, die Stellen zu identifizieren, in denen wir
        // definitiv NICHT rekursiv aufgerufen werden, und dann nach dem RELEASE zu warten...
#ifdef BUMP_WRITER_PRIORITY
        p_Lock->m_WriterPending = CLAIMRWLOCKWRITERPENDINGSIG;
#endif
        xSemaphoreTake(p_Lock->m_GroupSeparator,portMAX_DELAY);
#ifdef BUMP_WRITER_PRIORITY
        p_Lock->m_WriterPending = ~CLAIMRWLOCKWRITERPENDINGSIG;
#endif
    }
    else
    {
        // 1. Leser holt sich den lock und macht den Anderen die Tr auf
        if (INCREMENTTEST0(p_Lock))
        {
            xSemaphoreTake(p_Lock->m_GroupSeparator,portMAX_DELAY);
            p_Lock->m_ReaderSpinLock = 1;
        }
        else
        {
            volatile unsigned long aDum = 2;
        }
        while (!p_Lock->m_ReaderSpinLock) 
            vTaskDelay(2);  // FreeRTOS rundet nach unten ab...
    }
}

/** @brief Implements an RW lock claim. Callable in applications where there is only one R/W lock. 
 *
 * To study the behavior of wrong implementations of RW locks, you can have this function return immediately.
 *
 *  @param p_Mode: LOCK_WRITER or LOCK_READER
 *  @return none (blocks until the lock becomes or is available for the caller's purpose)
 */

void ClaimRwLock(unsigned short p_Mode)
{
    ClaimRwLockInternal(&g_SLM,p_Mode);  
}


/** @brief Implements an RW lock release. Internal function so the caller won't need to deal with data structures. 
 *
 *  @param p_Lock Pointer to a lock object
 *  @param p_Mode: LOCK_WRITER or LOCK_READER
 *  @return none 
 */

void ReleaseRWLockInternal(PRWLockManagement p_Lock,unsigned short p_Mode)
{
    if (p_Mode == LOCK_WRITER)
        xSemaphoreGive(p_Lock->m_GroupSeparator);
    else
    {
        if (DECREMENTTESTNEG(p_Lock))
        {
            // alle offenen Signalisierumgem wegwerfen. Das funktioniert genau deswegen, weil der einzige competitor (die andere reader task) nur im
            // claim Fall gegen uns laufen kann, also first claim gegen last release - und im first claim Fall mu der Gegner ja erst die globale Mutex 
            // wieder holen, die wir noch nicht freigegeben haben. Das klappt dann also...
            p_Lock->m_ReaderSpinLock = 0;
            xSemaphoreGive(p_Lock->m_GroupSeparator);
        }
        // wie oben beschrieben - um sicherzustellen, da ein writer nicht ausgelockt wird, mssen wir jeden nicht rekursiven Aufruf zu release nach dem
        // erfolgreichen release testen, ob wir die task davon abhalten mssen, sofort wieder den lock zu claimen.
#ifdef BUMP_WRITER_PRIORITY
        if (p_Mode == LOCK_READER_TERMINAL)
        {
            while (p_Lock->m_WriterPending == CLAIMRWLOCKWRITERPENDINGSIG)
                vTaskDelay(2);
        }
#endif        
    }
}

/** @brief Implements an RW lock release. Callable in applications where there is only one R/W lock. 
 *
 * To study the behavior of wrong implementations of RW locks, you can have this function return immediately.
 *
 *  @param p_Mode: LOCK_WRITER or LOCK_READER
 *  @return none 
 */

void ReleaseRWLock(unsigned short p_Mode)
{
    ReleaseRWLockInternal(&g_SLM,p_Mode);  
}

